# Step-by-Step Tutorial: Building Infinite Worlds

This tutorial takes you from zero to procedural world generation in small steps. Each step builds on the previous one.

---

## Step 1: A Single Box in Space

Start with the absolute minimum - one object in a 3D space.

```html
<!DOCTYPE html>
<html>
<head>
  <script src="https://aframe.io/releases/1.6.0/aframe.min.js"></script>
</head>
<body>
  <a-scene>
    <a-box position="0 1 -3" color="#4CC3D9"></a-box>
    <a-sky color="#ECECEC"></a-sky>
  </a-scene>
</body>
</html>
```

**What you learned:**
- `<a-scene>` contains everything
- `<a-box>` creates a 3D cube
- `position="x y z"` places objects in space
- Negative Z is "forward" from the starting view

---

## Step 2: Multiple Objects with Different Properties

```html
<!DOCTYPE html>
<html>
<head>
  <script src="https://aframe.io/releases/1.6.0/aframe.min.js"></script>
</head>
<body>
  <a-scene>
    <!-- Different shapes -->
    <a-box position="-2 1 -4" color="#FF6B6B"></a-box>
    <a-sphere position="0 1 -4" radius="0.8" color="#4ECDC4"></a-sphere>
    <a-cylinder position="2 1 -4" radius="0.5" height="1.5" color="#FFE66D"></a-cylinder>
    
    <!-- Ground -->
    <a-plane rotation="-90 0 0" width="10" height="10" color="#7BC8A4"></a-plane>
    
    <a-sky color="#87CEEB"></a-sky>
  </a-scene>
</body>
</html>
```

**What you learned:**
- Different primitive shapes: box, sphere, cylinder, plane
- Each shape has unique properties (radius, height, etc.)
- Rotation uses degrees around X, Y, Z axes

---

## Step 3: Adding JavaScript - Creating Objects with Code

```html
<!DOCTYPE html>
<html>
<head>
  <script src="https://aframe.io/releases/1.6.0/aframe.min.js"></script>
</head>
<body>
  <a-scene>
    <a-entity id="world"></a-entity>
    <a-plane rotation="-90 0 0" width="20" height="20" color="#7BC8A4"></a-plane>
    <a-sky color="#87CEEB"></a-sky>
  </a-scene>

  <script>
    // Wait for scene to load
    document.querySelector('a-scene').addEventListener('loaded', function() {
      const world = document.getElementById('world');
      
      // Create 10 boxes with JavaScript
      for (let i = 0; i < 10; i++) {
        const box = document.createElement('a-box');
        box.setAttribute('position', `${i * 2 - 9} 0.5 -5`);
        box.setAttribute('color', '#4CC3D9');
        world.appendChild(box);
      }
    });
  </script>
</body>
</html>
```

**What you learned:**
- Create elements with `document.createElement()`
- Set attributes with `setAttribute()`
- Add to scene with `appendChild()`
- Use loops to create multiple objects

---

## Step 4: Random Placement

```html
<!DOCTYPE html>
<html>
<head>
  <script src="https://aframe.io/releases/1.6.0/aframe.min.js"></script>
</head>
<body>
  <a-scene>
    <a-entity id="world"></a-entity>
    <a-plane rotation="-90 0 0" width="50" height="50" color="#7BC8A4"></a-plane>
    <a-sky color="#87CEEB"></a-sky>
    <a-entity camera look-controls wasd-controls position="0 1.6 0"></a-entity>
  </a-scene>

  <script>
    document.querySelector('a-scene').addEventListener('loaded', function() {
      const world = document.getElementById('world');
      const colors = ['#FF6B6B', '#4ECDC4', '#FFE66D', '#95E1D3', '#F38181'];
      
      // Create 50 randomly placed boxes
      for (let i = 0; i < 50; i++) {
        const box = document.createElement('a-box');
        
        // Random position
        const x = (Math.random() - 0.5) * 40;
        const z = (Math.random() - 0.5) * 40;
        
        // Random color
        const color = colors[Math.floor(Math.random() * colors.length)];
        
        // Random size
        const size = 0.3 + Math.random() * 0.7;
        
        box.setAttribute('position', `${x} ${size/2} ${z}`);
        box.setAttribute('color', color);
        box.setAttribute('width', size);
        box.setAttribute('height', size);
        box.setAttribute('depth', size);
        
        world.appendChild(box);
      }
    });
  </script>
</body>
</html>
```

**What you learned:**
- `Math.random()` gives 0.0 to 0.999...
- Multiply and offset to get desired range
- Pick randomly from arrays

---

## Step 5: Seeded Random (Reproducible Randomness)

```html
<!DOCTYPE html>
<html>
<head>
  <script src="https://aframe.io/releases/1.6.0/aframe.min.js"></script>
</head>
<body>
  <a-scene>
    <a-entity id="world"></a-entity>
    <a-plane rotation="-90 0 0" width="50" height="50" color="#7BC8A4"></a-plane>
    <a-sky color="#87CEEB"></a-sky>
    <a-entity camera look-controls wasd-controls position="0 1.6 0"></a-entity>
  </a-scene>

  <script>
    // THE KEY TO PROCEDURAL GENERATION
    // Same seed = same "random" numbers every time
    function seededRandom(seed) {
      const x = Math.sin(seed * 9999) * 9999;
      return x - Math.floor(x);
    }
    
    // Try changing this number - each seed creates a different world!
    const WORLD_SEED = 12345;
    
    document.querySelector('a-scene').addEventListener('loaded', function() {
      const world = document.getElementById('world');
      const colors = ['#FF6B6B', '#4ECDC4', '#FFE66D', '#95E1D3', '#F38181'];
      
      for (let i = 0; i < 50; i++) {
        const box = document.createElement('a-box');
        
        // Use seeded random instead of Math.random()
        // Each object gets its own seed derived from the world seed
        const x = (seededRandom(WORLD_SEED + i * 3) - 0.5) * 40;
        const z = (seededRandom(WORLD_SEED + i * 3 + 1) - 0.5) * 40;
        const colorIndex = Math.floor(seededRandom(WORLD_SEED + i * 3 + 2) * colors.length);
        const size = 0.3 + seededRandom(WORLD_SEED + i * 3 + 3) * 0.7;
        
        box.setAttribute('position', `${x} ${size/2} ${z}`);
        box.setAttribute('color', colors[colorIndex]);
        box.setAttribute('width', size);
        box.setAttribute('height', size);
        box.setAttribute('depth', size);
        
        world.appendChild(box);
      }
    });
  </script>
</body>
</html>
```

**What you learned:**
- Seeded random is deterministic - same input = same output
- Change the seed to get a different (but reproducible) world
- This is how multiplayer games ensure everyone sees the same world

---

## Step 6: Chunks - Dividing the World

```html
<!DOCTYPE html>
<html>
<head>
  <script src="https://aframe.io/releases/1.6.0/aframe.min.js"></script>
  <style>
    #info { position: fixed; top: 10px; left: 10px; color: white; 
            font-family: monospace; background: rgba(0,0,0,0.5); padding: 10px; }
  </style>
</head>
<body>
  <div id="info">Chunks loaded: <span id="count">0</span></div>
  <a-scene>
    <a-entity id="world"></a-entity>
    <a-sky color="#87CEEB"></a-sky>
    <a-light type="ambient" intensity="0.5"></a-light>
    <a-entity id="player" camera look-controls wasd-controls position="0 1.6 0"></a-entity>
  </a-scene>

  <script>
    function seededRandom(seed) {
      const x = Math.sin(seed * 9999) * 9999;
      return x - Math.floor(x);
    }
    
    const WORLD_SEED = 42;
    const CHUNK_SIZE = 20;
    const chunks = new Map();
    
    function createChunk(chunkX, chunkZ) {
      const key = `${chunkX},${chunkZ}`;
      if (chunks.has(key)) return;
      
      const chunk = document.createElement('a-entity');
      const worldX = chunkX * CHUNK_SIZE;
      const worldZ = chunkZ * CHUNK_SIZE;
      chunk.setAttribute('position', `${worldX} 0 ${worldZ}`);
      
      // Each chunk has its own seed based on position
      const chunkSeed = WORLD_SEED + chunkX * 1000 + chunkZ;
      
      // Ground for this chunk
      const ground = document.createElement('a-plane');
      ground.setAttribute('rotation', '-90 0 0');
      ground.setAttribute('width', CHUNK_SIZE);
      ground.setAttribute('height', CHUNK_SIZE);
      ground.setAttribute('position', `${CHUNK_SIZE/2} 0 ${-CHUNK_SIZE/2}`);
      // Vary color slightly per chunk
      const g = 100 + Math.floor(seededRandom(chunkSeed) * 50);
      ground.setAttribute('color', `rgb(60, ${g}, 60)`);
      chunk.appendChild(ground);
      
      // Random trees in this chunk
      const numTrees = Math.floor(seededRandom(chunkSeed + 1) * 5) + 1;
      for (let i = 0; i < numTrees; i++) {
        const treeSeed = chunkSeed + 100 + i * 10;
        const tx = seededRandom(treeSeed) * CHUNK_SIZE;
        const tz = -seededRandom(treeSeed + 1) * CHUNK_SIZE;
        const height = 2 + seededRandom(treeSeed + 2) * 3;
        
        // Trunk
        const trunk = document.createElement('a-cylinder');
        trunk.setAttribute('position', `${tx} ${height * 0.3} ${tz}`);
        trunk.setAttribute('radius', '0.2');
        trunk.setAttribute('height', height * 0.6);
        trunk.setAttribute('color', '#8B4513');
        chunk.appendChild(trunk);
        
        // Foliage
        const foliage = document.createElement('a-cone');
        foliage.setAttribute('position', `${tx} ${height * 0.7} ${tz}`);
        foliage.setAttribute('radius-bottom', '1');
        foliage.setAttribute('height', height * 0.6);
        foliage.setAttribute('color', '#228B22');
        chunk.appendChild(foliage);
      }
      
      document.getElementById('world').appendChild(chunk);
      chunks.set(key, chunk);
      document.getElementById('count').textContent = chunks.size;
    }
    
    function updateChunks() {
      const player = document.getElementById('player');
      const pos = player.getAttribute('position');
      
      // Which chunk is the player in?
      const playerChunkX = Math.floor(pos.x / CHUNK_SIZE);
      const playerChunkZ = Math.floor(pos.z / CHUNK_SIZE);
      
      // Create chunks in a 3x3 area around player
      for (let dx = -1; dx <= 1; dx++) {
        for (let dz = -1; dz <= 1; dz++) {
          createChunk(playerChunkX + dx, playerChunkZ + dz);
        }
      }
    }
    
    document.querySelector('a-scene').addEventListener('loaded', function() {
      updateChunks();
      setInterval(updateChunks, 500);
    });
  </script>
</body>
</html>
```

**What you learned:**
- Divide infinite space into finite chunks
- Generate chunks on-demand as player moves
- Each chunk's content is determined by its position + world seed
- Same chunk position always generates same content

---

## Step 7: The Final Piece - Removing Old Chunks

Add this to the `updateChunks` function to make the world truly infinite:

```javascript
// Remove chunks that are too far away
for (const [key, chunk] of chunks) {
  const [cx, cz] = key.split(',').map(Number);
  const distance = Math.max(
    Math.abs(cx - playerChunkX),
    Math.abs(cz - playerChunkZ)
  );
  
  if (distance > 3) {
    chunk.parentNode.removeChild(chunk);
    chunks.delete(key);
  }
}
```

**What you learned:**
- Memory is finite; remove what you don't need
- The world feels infinite even though only nearby chunks exist
- In "Infinite Forward," we make this one-directional: only remove behind

---

## Congratulations!

You now understand the core concepts behind procedural world generation:

1. **Primitives**: Basic 3D shapes
2. **JavaScript creation**: Building scenes with code
3. **Randomness**: Creating variety
4. **Seeded random**: Making it reproducible
5. **Chunks**: Dividing infinite space
6. **Streaming**: Loading/unloading as needed

The "Infinite Forward" project combines all these with:
- Forward-only movement (no going back)
- Sky changes (atmosphere shifts)
- Messages (narrative elements)
- The void (visual representation of impermanence)

---

## Next Steps

1. Open `learn-procedural.html` to see these concepts in action
2. Open `index.html` to experience the full Infinite Forward
3. Read the code comments to understand how it all fits together
4. Modify, break, fix, and make it your own!

*The best way to learn is to build.*
